package loquebot.memory;

import java.util.HashMap;
import java.util.ArrayList;
import java.util.logging.Logger;

import cz.cuni.pogamut.Client.AgentBody;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;

import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.DeleteFromBatch;

import loquebot.Main;
import loquebot.util.LoqueListener;

/**
 * Responsible for listening to the messages and managing other map players.
 *
 * PLAYER, ID 153, UnrealID DM-0-Medium.GBxPlayer1, name Player, team 255,
 * reachable false, visible true,
 * location 2022.08,-60.72,49.13, rotation 0.0,29919.0,0.0, velocity 0.0,0.0,0.0
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoquePlayers
{
    /**
     * Retreives last known info about given player.
     * @param ID Player ID to be retreived.
     * @return Last known player info; or null upon none.
     * @note Returns copy of the player info to prevent outer modifications.
     */
    public synchronized Player getPlayer (int ID)
    {
        // retreive player
        Player player = allPlayersIDs.get (ID);
        // return cloned player or null, upon none
        return (player == null) ? null : (Player) player.clone ();
    }

    /**
     * Retreives last known info about given player.
     * @param UnrealID Player UnrealID to be retreived.
     * @return Last known player info; or null upon none.
     * @note Returns copy of the player info to prevent outer modifications.
     */
    public synchronized Player getPlayer (String UnrealID)
    {
        // retreive player
        Player player = allPlayersUnrealIDs.get (UnrealID);
        // return cloned player or null, upon none
        return (player == null) ? null : (Player) player.clone ();
    }

    /*========================================================================*/

    /**
     * Retreives info about given player, but only it the player is visible.
     * @param ID Player ID to be retreived.
     * @return Player info; or null upon none or not visible.
     * @note Returns copy of the player info to prevent outer modifications.
     */
    public synchronized Player getVisiblePlayer (int ID)
    {
        // retreive player
        Player player = visiblePlayersIDs.get (ID);
        // return cloned player or null, upon none
        return (player == null) ? null : (Player) player.clone ();
    }

    /**
     * Retreives info about given player, but only it the player is visible.
     * @param UnrealID Player UnrealID to be retreived.
     * @return Player info; or null upon none or not visible.
     * @note Returns copy of the player info to prevent outer modifications.
     */
    public synchronized Player getVisiblePlayer (String UnrealID)
    {
        // retreive player
        Player player = visiblePlayersUnrealIDs.get (UnrealID);
        // return cloned player or null, upon none
        return (player == null) ? null : (Player) player.clone ();
    }

    /*========================================================================*/

    /**
     * Tells, whether the agent sees any other players.
     * @return True, if other players are visible; false otherwise.
     */
    public synchronized boolean hasVisiblePlayers ()
    {
        // check visible map size
        return (visiblePlayersIDs.size () > 0);
    }

    /**
     * Tells, whether the agent sees any enemies.
     * @return True, if any enemies are visible; false otherwise.
     */
    public synchronized boolean hasVisibleEnemies ()
    {
        // check visible map size
        return (visibleEnemiesIDs.size () > 0);
    }

    /*========================================================================*/

    /**
     * Retreives list of all visible players.
     * @return List of all visible players.
     * @note Returns copies of the player infos to prevent outer modifications.
     */
    public synchronized ArrayList<Player> getVisiblePlayers ()
    {
        ArrayList<Player> result = new ArrayList<Player> (visiblePlayersIDs.size ());
        // make deep clone of the players
        for (Player p : visiblePlayersIDs.values ())
            result.add((Player) p.clone ());
        // return created list
        return result;
    }

    /**
     * Retreives list of all visible enemies.
     * @return List of all visible enemies.
     * @note Returns copies of the enemy infos to prevent outer modifications.
     */
    public synchronized ArrayList<Player> getVisibleEnemies ()
    {
        ArrayList<Player> result = new ArrayList<Player> (visibleEnemiesIDs.size ());
        // make deep clone of the players
        for (Player p : visibleEnemiesIDs.values ())
            result.add((Player) p.clone ());
        // return created list
        return result;
    }

    /*========================================================================*/
    /*========================================================================*/
    /*========================================================================*/

    /** All seen players mapped by their IDs. */
    private HashMap<Integer, Player> allPlayersIDs = new HashMap<Integer, Player> ();
    /** All seen players mapped by their UnrealIDs. */
    private HashMap<String, Player> allPlayersUnrealIDs = new HashMap<String, Player> ();

    /** Visible players mapped by their IDs. */
    private HashMap<Integer, Player> visiblePlayersIDs = new HashMap<Integer, Player> ();
    /** Visible players mapped by their UnrealIDs. */
    private HashMap<String, Player> visiblePlayersUnrealIDs = new HashMap<String, Player> ();

    /** Visible enemies mapped by their IDs. */
    private HashMap<Integer, Player> visibleEnemiesIDs = new HashMap<Integer, Player> ();
    /** Visible enemies mapped by their UnrealIDs. */
    private HashMap<String, Player> visibleEnemiesUnrealIDs = new HashMap<String, Player> ();

    /*========================================================================*/

    /**
     * Adds new player into visible players.
     * @param player Player message to be relied upon.
     */
    private synchronized void addPlayer (Player player)
    {
        // do we have this player in visibles?
        if (null == visiblePlayersIDs.get(player.ID))
            log.config("Memory.Player.addPlayer(): see new player " + player.UnrealID);

        // add this player into all maps
        allPlayersIDs.put (player.ID, player);
        allPlayersUnrealIDs.put (player.UnrealID, player);

        // is it visible? well, i guess so..
        if (player.visible)
        {
            visiblePlayersIDs.put (player.ID, player);
            visiblePlayersUnrealIDs.put (player.UnrealID, player);

            // is it an enemy?
            if (memory.isEnemy(player))
            {
                // add this player into enemy maps as well
                visibleEnemiesIDs.put (player.ID, player);
                visibleEnemiesUnrealIDs.put (player.UnrealID, player);
            }
        }
    }

    /**
     * Removes given player from visible players.
     * @param msg Delete message to be relied upon.
     */
    private synchronized void removePlayer (DeleteFromBatch msg)
    {
        // is it interesting?
        if (msg.msgType != MessageType.PLAYER)
            return;

        // do we have this player in our lists?
        Player player = allPlayersIDs.get(msg.ID);
        if (player != null)
        {
            // remove and check if removed
            if (null != visiblePlayersIDs.remove (player.ID))
            {
                // remove from others..
                visiblePlayersUnrealIDs.remove (player.UnrealID);

                // remove and check if removed
                if (null != visibleEnemiesIDs.remove (player.ID))
                {
                    // remove from others..
                    visibleEnemiesUnrealIDs.remove (player.UnrealID);
                }

                log.config("Memory.Player.removePlayer(): lost seen player " + player.UnrealID);
            }
        }
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Info about other player.
         * @param msg Message to handle.
         */
        private void msgPlayer (Player msg)
        {
            addPlayer (msg);
        }

        /**
         * Info about loosing things from sight.
         * @param msg Message to handle.
         */
        private void msgDeleteFromBatch (DeleteFromBatch msg)
        {
            removePlayer (msg);
        }

        /*========================================================================*/

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case PLAYER:
                    msgPlayer ((Player) msg);
                    return;
                case DELETE_FROM_BATCH:
                    msgDeleteFromBatch ((DeleteFromBatch) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener (this, MessageType.PLAYER);
            body.addTypedRcvMsgListener (this, MessageType.DELETE_FROM_BATCH);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /** Agent's main. */
    protected Main main;
    /** Loque memory. */
    protected LoqueMemory memory;
    /** Agent's body. */
    protected AgentBody body;
    /** Agent's log. */
    protected Logger log;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoquePlayers (Main main, LoqueMemory memory)
    {
        // setup reference to agent
        this.main = main;
        this.memory = memory;
        this.body = main.getBody ();
        this.log = main.getLogger ();

        // create listener
        this.listener = new Listener ();
    }
}